/*++

	Copyright (c) 1997,  Dr. Johannes Heidenhain GmbH

	Module Name:	I2C.C
	
	Version 1.0:	18.12.1997

	Abstract:		Windows DLL for IK 121

	Notes:			This DLL provides IK 121 card functions for IK 121.
					IK 121 hardware is accessed through IK121Drv.Sys.

	Function:		Low level functions for I2C POT and EEPROM.
					The values of the amplitude and phase shift of the encoder signals
					are stored in register 0 of the EEPROM.
					The offset values of the encoder signals are stored in
					registers 2 and 3.

					Pot 0 - Register 0 :  phase compensation of axis 1
					Pot 0 - Register 1 :  not used
					Pot 0 - Register 2 :  offset  0 degree (lo) of axis 1	 
					Pot 0 - Register 3 :  offset  0 degree (hi) of axis 1

					Pot 1 - Register 0 :  amplitude compensation of axis 1
					Pot 1 - Register 1 :  not used
					Pot 1 - Register 2 :  offset 90 degree (lo) of axis 1
					Pot 1 - Register 3 :  offset 90 degree (hi) of axis 1

					Pot 2 - Register 0 :  amplitude compensation of axis 2
					Pot 2 - Register 1 :  not used
					Pot 2 - Register 2 :  offset  0 degree (lo) of axis 2
					Pot 2 - Register 3 :  offset  0 degree (hi) of axis 2

					Pot 3 - Register 0 :  phase compensation of axis 2
					Pot 3 - Register 1 :  not used
					Pot 3 - Register 2 :  offset 90 degree (lo) of axis 2
					Pot 3 - Register 3 :  offset 90 degree (hi) of axis 2

--*/

#include "DLLGlob.h"                   


extern USHORT RegW;
extern BOOL SetI2C (USHORT Card, USHORT Data);


//////////////////////////////////////////////////////
// SetI2C: Sets clock and data signal of I2C interface
//////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKSetI2C(USHORT Card, BOOL SCL, BOOL SDA)
{
  	USHORT Buffer;

	Buffer = Ram_LatchReg[Card<<1+1];
	if(!SCL) Buffer |= 0x8000;
	if(!SDA) Buffer |= 0x0800;
	return SetI2C(Card, Buffer);
}


// Set clock and data signal
////////////////////////////
BOOL Scl0_Sda0(USHORT Card) { return IKSetI2C(Card, 0, 0); }
BOOL Scl0_Sda1(USHORT Card) { return IKSetI2C(Card, 0, 1); }
BOOL Scl1_Sda0(USHORT Card) { return IKSetI2C(Card, 1, 0); }
BOOL Scl1_Sda1(USHORT Card) { return IKSetI2C(Card, 1, 1); }



// StartI2C: Starts transmission on I2C (Send start bit)
////////////////////////////////////////////////////////
BOOL StartI2C(USHORT Card)
{												// SCL	SDA	
	if (!Scl1_Sda1(Card)) return FALSE;			//  1	 1
	if (!Scl1_Sda0(Card)) return FALSE;			//	1	 0
	return Scl0_Sda0(Card);						//  0	 0
}


// OutI2C: Sends a bit on I2C
/////////////////////////////
BOOL OutI2C (USHORT Card, BOOL Bit)
{
	if (Bit)	// Bit = TRUE -> SDA=1
	{											// SCL	SDA	
		if (!Scl0_Sda1(Card)) return FALSE;		//	0	 1
		if (!Scl1_Sda1(Card)) return FALSE;		//	1	 1
		if (!Scl0_Sda1(Card)) return FALSE;		//	0	 1
	}
	else 		// Bit = FALSE -> SDA=0 
	{											// SCL	SDA	
		if (!Scl0_Sda0(Card)) return FALSE;		//  0	 0
		if (!Scl1_Sda0(Card)) return FALSE;		//  1	 0
		if (!Scl0_Sda0(Card)) return FALSE;		//  0	 0
	}
	return TRUE;
}


// InI2C: Reads a bit from I2C
//////////////////////////////
BOOL InI2C (USHORT Card, BOOL* Bit)
{																					// SCL	SDA	
	if (!Scl0_Sda1(Card)) return FALSE;												//  0	 1
	if (!Scl1_Sda1(Card)) return FALSE;												//  1	 1
	if (!IKInputW((USHORT)(Card*2+1), ContrReg, (PUSHORT)Bit)) return FALSE;		// Read data bit 
//	if (!Scl0_Sda1(Card)) return FALSE;												//  0	 1
	*Bit = (*Bit) & 0x0020;															// Mask data bit
	return TRUE;
}


// StopI2C: Stops transmission on I2C (Send stop bit)
/////////////////////////////////////////////////////
BOOL StopI2C(USHORT Card)
{											// SCL	SDA	
	if (!Scl0_Sda0(Card)) return FALSE;		//  0	 0
	if (!Scl1_Sda0(Card)) return FALSE;		//  1	 0
	if (!Scl1_Sda1(Card)) return FALSE;		//  1	 1
	if (!Scl1_Sda1(Card)) return FALSE;		//  1	 1
	return Scl0_Sda1(Card);					//  0	 1
}


// Write8I2C: Sends a byte(8 bit) to I2C
////////////////////////////////////////
BOOL Write8I2C (USHORT Card, BYTE Data)
{
	BYTE Mask;

	for (Mask=0x80; Mask!=0; Mask=Mask>>1) 
		if (!OutI2C (Card, (Data & Mask))) return FALSE;
	return TRUE;
}


// Read8I2C: Reads a byte(8-bit) from I2C
/////////////////////////////////////////
BOOL Read8I2C (USHORT Card, BYTE* Data)
{
	USHORT i;
	BOOL   Bit;

	*Data=0;
	for (i=0; i<8; i++) 
	{
		*Data = *Data << 1;
		if (!InI2C (Card, &Bit)) return FALSE;
		if (Bit) *Data = *Data | 0x01;
	}
	return TRUE;
}


// PollPoti: Waits until pot is ready, and starts transmission
//////////////////////////////////////////////////////////////
BOOL PollPoti (USHORT Card)
{
	BOOL  Bit;
	USHORT TimeOut = 0;
	
	do
	{
		if (!StartI2C  (Card))       return FALSE;		// Start bit 
		if (!Write8I2C (Card, 0x5F)) return FALSE;		// Pot ID and address

		if (!InI2C(Card, &Bit)) return FALSE;			// Read acknowledge
		if (Bit)										// No acknowledge
		{
			if (!StopI2C(Card)) return FALSE;			// Stop bit
		   	TimeOut += 1;								// Increment timeout counter
			Sleep (1);									// Wait 1 ms
		}
	}
	while ( Bit && (TimeOut < 100) );					// Acknowledge from pot or timeout
	if (TimeOut<100) return TRUE; else return FALSE;
}


// PollRom: Waits until EEPROM is ready, and starts transmission
////////////////////////////////////////////////////////////////
BOOL PollRom (USHORT Card, BYTE DevInfo)
{
	BOOL Bit;
	BYTE TimeOut = 0;
	
	do
	{
		if (!StartI2C  (Card))				return FALSE;	// Start bit                       
		if (!Write8I2C (Card, DevInfo))		return FALSE;	// EEPROM ID, address and read/write
		if (!InI2C(Card, &Bit))				return FALSE;	// Read acknowledge                
		if (Bit)
		{	
			if (!StopI2C(Card))				return FALSE;	// Stop bit
			TimeOut += 1;									// Increment timeout counter                                   
			Sleep (1);										// Wait 1 ms
		}
	}
	while ( Bit && (TimeOut < 100) );						// Acknowledge from EEPROM or timeout
	if (TimeOut<100) return TRUE; else return FALSE;	
}														                                   
														                                   

// PotiWrite2: Sends a 2-byte instruction to pot
////////////////////////////////////////////////
BOOL PotiWrite2 (USHORT Card, BYTE Instruct, BYTE Poti, BYTE Reg)
{
	BOOL Bit;

	if (!PollPoti  (Card))											return FALSE;		// Wait until pot ready
	if (!Write8I2C (Card, (BYTE)(Instruct<<4 | Poti<<2 | Reg )))	return FALSE;		// Pot command
	if (!InI2C(Card, &Bit))											return FALSE;		// Read acknowledge
	if (!StopI2C(Card))												return FALSE;		// Stop bit
	if (Bit) return FALSE; else return TRUE;
}


// PotiWrite2: Sends a 3-byte instruction to pot
////////////////////////////////////////////////
BOOL PotiWrite3 (USHORT Card, BYTE Instruct, BYTE Poti, BYTE Reg, BYTE Data)
{
	BOOL Bit;

	if (!PollPoti  (Card))											return FALSE;		// Wait until pot ready
	if (!Write8I2C (Card, (BYTE)(Instruct<<4 | Poti<<2 | Reg )))	return FALSE;		// Pot command
	if (!InI2C     (Card, &Bit)) 									return FALSE;		// Read acknowledge
	if (!Write8I2C (Card, Data))									return FALSE;		// Write Data
	if (!InI2C     (Card, &Bit)) 									return FALSE;		// Read acknowledge
	return StopI2C(Card);														// Stop bit
}


// PotiRead: Reads a byte from pot
//////////////////////////////////
BOOL PotiRead(USHORT Card, USHORT Instruct, USHORT Poti, USHORT Reg, BYTE* pData)
{
	BOOL Bit;
	
	if (!PollPoti  (Card))											return FALSE;		// Wait until pot ready
	if (!Write8I2C (Card, (BYTE)(Instruct<<4 | Poti<<2 | Reg )))	return FALSE;		// Poti command
	if (!InI2C     (Card, &Bit)) 									return FALSE;		// Read acknowledge
	if (Bit)														return FALSE;		// No acknowledge
	if (!Read8I2C  (Card, pData))									return FALSE;		// Read Data
	if (!OutI2C    (Card, FALSE))									return FALSE;		// Acknowledge
	if (!StopI2C   (Card))											return FALSE;		// Stop bit
}


//////////////////////////////////////////////////////////////
// IKReadPhase: Reads phase compensation poti of Axis to pData
//////////////////////////////&&//////////////////////////////
DLLEXPORT BOOL WINAPI IKReadPhase (USHORT Axis, BYTE* pData)
{
	USHORT Card = Axis>>1;

	if (Axis>15) return FALSE;
	if (!(Axis & 0x01)) return PotiRead (Card, 0x09, 0x00, 0x00, pData);		// Read WCR pot 0
	else				return PotiRead (Card, 0x09, 0x03, 0x00, pData);		// Read WCR pot 3
}


/////////////////////////////////////////////////////////////
// IKWritePhase: Write Data to phase compensation pot of Axis
/////////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKWritePhase (USHORT Axis, BYTE Data)
{
	USHORT Card = Axis>>1;

	if (Axis>15) return FALSE;
	if (!(Axis & 0x01))
	{
		return PotiWrite3 (Card, 0x0A, 0x00, 0x00, Data);					// Write Data to WCR pot 0 
	}
	else
	{
		return PotiWrite3 (Card, 0x0A, 0x03, 0x00, Data);					// Write Data to WCR pot 3
	}
}


////////////////////////////////////////////////////////////////
// IKLoadPhase: Reads stored phase compensation of Axis to pData
////////////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKLoadPhase (USHORT Axis, BYTE* pData)
{
	USHORT Card = Axis>>1;

	if (Axis>15) return FALSE;
	if (!(Axis & 0x01)) return PotiRead (Card, 0x0B, 0x00, 0x00, pData);		// Read register 0 of pot 0
	else				return PotiRead (Card, 0x0B, 0x03, 0x00, pData);		// Read register 0 of pot 3
}


///////////////////////////////////////////////////////////////
// IKReadAmp: Reads amplitude compensation pot of Axis to pData
///////////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKReadAmp (USHORT Axis, BYTE* pData)
{
	USHORT Card = Axis>>1;

	if (Axis>15) return FALSE;
	if (!(Axis & 0x01)) return PotiRead(Card, 0x09, 0x01, 0x00, pData);		// Read WCR pot 1
	else	  			return PotiRead(Card, 0x09, 0x02, 0x00, pData);		// Read WCR pot 2
}


///////////////////////////////////////////////////////////////
// IKWriteAmp: Write Data to amplitude compensation pot of Axis
///////////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKWriteAmp (USHORT Axis, BYTE Data)
{
	USHORT Card = Axis>>1;

	if (Axis>15) return FALSE;
	if (!(Axis & 0x01))
	{
		return PotiWrite3 (Card, 0x0A, 0x01, 0x00, Data);					// Write Data to WCR pot 1
	}
	else
	{
		return PotiWrite3 (Card, 0x0A, 0x02, 0x00, Data);					// Write Data to WCR pot 2
	}
}


//////////////////////////////////////////////////////////////////
// IKLoadAmp: Reads stored amplitude compensation of Axis to pData
//////////////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKLoadAmp (USHORT Axis, BYTE* pData)
{
	USHORT Card = Axis>>1;

	if (Axis>15) return FALSE;
	if (!(Axis & 0x01)) return PotiRead (Card, 0x0B, 0x01, 0x00, pData);		// Read register 0 of pot 1
	else				return PotiRead (Card, 0x0B, 0x02, 0x00, pData);		// Read register 0 of pot 2
}



////////////////////////////////////////////////////////
// IKReadOffset: Read offset compensation values of Axis
////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKReadOffset (USHORT Axis, SHORT* Ofs0, SHORT* Ofs90)
{
	*Ofs0  = Ram_OfsReg0 [Axis];						// Load Ofs0  with shadow register
	*Ofs90 = Ram_OfsReg90[Axis];						// Load Ofs90 with shadow register
	
	if (*Ofs0  & 0x0040) *Ofs0  = *Ofs0  | 0xFF80;		// Sign extend Ofs0
	if (*Ofs90 & 0x0040) *Ofs90 = *Ofs90 | 0xFF80;		// Sign extend Ofs90
	return TRUE;
}


///////////////////////////////////////////////////////////
// IKWriteOffset: Writes offset compensation values of Axis
///////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKWriteOffset (USHORT Axis, SHORT Ofs0, SHORT Ofs90)
{
	USHORT TimeOut=0;

	if (Ofs0  >  63) Ofs0  =  63;		// Limit Ofs0  to +/-63
	if (Ofs0  < -63) Ofs0  = -63;
	if (Ofs90 >  63) Ofs90 =  63;		// Limit Ofs90 to +/-63
	if (Ofs90 < -63) Ofs90 = -63;
	do
	{
		if (!IKInputW ( Axis, StatReg3, &RegW )) return FALSE;		// Wait until offset register ready
		TimeOut += 1;
	}
	while ( (RegW & 0x0060) && (TimeOut<100) );
	if (TimeOut<100)
	{
	  if (!IKOutput(Axis, OfsReg0,  Ofs0 ))		 return FALSE;	 	// Write Ofs0  to offset  0 register
	}
	else return FALSE;
	return IKOutput(Axis, OfsReg90, Ofs90);							// Write Ofs90 to offset 90 register
}
 

//////////////////////////////////////////////////////////////////
// IKLoadOffset: Read offset compensation values from pot register
//////////////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKLoadOffset (USHORT Axis, SHORT* Ofs0, SHORT* Ofs90)
{
	BYTE hi,lo;
	USHORT Card = Axis>>1;

	if (Axis>15) return FALSE;
	if (!(Axis & 0x01))
	{
		// Load offset 0  degree from pot 0 register 2/3
		if (!PotiRead (Card, 0x0B, 0x00, 0x02, &lo)) return FALSE;		// Read pot 0 register 2
		if (!PotiRead (Card, 0x0B, 0x00, 0x03, &hi)) return FALSE;		// Read pot 0 register 3
		*Ofs0 = (SHORT)((hi<<4) | (lo & 0x0F));
		

		// Load offset 90 degree from pot 1 register 2/3
		if (!PotiRead (Card, 0x0B, 0x01, 0x02, &lo)) return FALSE;		// Read pot 1 register 2	
		if (!PotiRead (Card, 0x0B, 0x01, 0x03, &hi)) return FALSE;		// Read pot 1 register 3
		*Ofs90 = (SHORT)((hi<<4) | (lo & 0x0F));
	}
	else
	{
		// Load offset 0  degree from pot 2 register 2/3
		if (!PotiRead (Card, 0x0B, 0x02, 0x02, &lo)) return FALSE;		// Read pot 2 register 2
		if (!PotiRead (Card, 0x0B, 0x02, 0x03, &hi)) return FALSE;		// Read pot 2 register 3
		*Ofs0 = (SHORT)((hi<<4) | (lo & 0x0F));

		// Load offset 90 degree from pot 3 register 3/3
		if (!PotiRead (Card, 0x0B, 0x03, 0x02, &lo)) return FALSE;		// Read pot 3 register 2
		if (!PotiRead (Card, 0x0B, 0x03, 0x03, &hi)) return FALSE;		// Read pot 3 register 3
		*Ofs90 = (SHORT)((hi<<4) | (lo & 0x0F));
	}
	if (*Ofs0  & 0x0040) *Ofs0  = *Ofs0  | 0xFF80;						// Sign extend
	if (*Ofs90 & 0x0040) *Ofs90 = *Ofs90 | 0xFF80;						// Sign extend
	return TRUE;
}


/////////////////////////////////////////////////////////////////////////
// IKStore: Saves phase, amplitude and offset compensation values of Axis
/////////////////////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKStore (USHORT Axis)
{
	BYTE lo, hi;
	BYTE Ofs0, Ofs90;

	USHORT Card = Axis>>1;

	if (Axis>15) return FALSE;

	Ofs0  = (BYTE)(Ram_OfsReg0 [Axis] & 0xFF);
	Ofs90 = (BYTE)(Ram_OfsReg90[Axis] & 0xFF);

	if (!(Axis & 0x01))
	{
		if (!PotiWrite2 (Card, 0x0E, 0x00, 0x00))	  return FALSE;		// Write WCR(phase)     pot 0 to register 0
		if (!PotiWrite2 (Card, 0x0E, 0x01, 0x00))	  return FALSE;		// Write WCR(amplitude) pot 1 to register 0

		// Offset 0  degree axis 1 saved in pot 0 register 2/3
		lo =  Ofs0 & 0x0F;	hi = (Ofs0>>4)  & 0x0F;
		if (!PotiWrite3 (Card, 0x0C, 0x00, 0x02, lo)) return FALSE;		// Write Ofs0(lo)  to pot 0 register 2
		if (!PotiWrite3 (Card, 0x0C, 0x00, 0x03, hi)) return FALSE;		// Write Ofs0(hi)  to pot 0 rRegister 3

		// Offset 90 degree axis 1 saved in pot 1 register 2/3
		lo =  Ofs90	& 0x0F;	hi = (Ofs90>>4) & 0x0F;
		if (!PotiWrite3 (Card, 0x0C, 0x01, 0x02, lo)) return FALSE;		// Write Ofs90(lo) to pot 1 register 2
		if (!PotiWrite3 (Card, 0x0C, 0x01, 0x03, hi)) return FALSE;		// Write Ofs90(hi) to pot 1 register 3
	}
	else
	{
		if (!PotiWrite2 (Card, 0x0E, 0x02, 0x00))	  return FALSE;		// Write WCR(phase)     pot 0 to register 0
		if (!PotiWrite2 (Card, 0x0E, 0x03, 0x00))	  return FALSE;		// Write WCR(amplitude) pot 1 to register 0
		
		// Offset 0  degree axis 2 saved in pot 2 register 2/3
		lo =  Ofs0 & 0x0F;	hi = (Ofs0>>4)  & 0x0F;
		if (!PotiWrite3 (Card, 0x0C, 0x02, 0x02, lo)) return FALSE;		// Write Ofs0(lo)  to pot 2 register 2
		if (!PotiWrite3 (Card, 0x0C, 0x02, 0x03, hi)) return FALSE;		// Write Ofs0(hi)  to pot 2 register 3

		// Offset 90 degree axis 2 saved in Poti 3 Register 2/3
		lo =  Ofs90 & 0x0F;	hi = (Ofs90>>4) & 0x0F;
		if (!PotiWrite3 (Card, 0x0C, 0x03, 0x02, lo)) return FALSE;		// Write Ofs90(lo) to pot 3 register 2
		if (!PotiWrite3 (Card, 0x0C, 0x03, 0x03, hi)) return FALSE;		// Write Ofs90(hi) to pot 3 register 3
	}
	return TRUE;
}


///////////////////////////////////////////////////////////////////////////////////
// IKDefault: Set default value for phase, amplitude and offset compensation values
///////////////////////////////////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKDefault (USHORT Axis)
{
	USHORT Card = Axis>>1;

	if (Axis>15) return FALSE;

	// Poti in neutral position 
	if (!IKWritePhase  (Axis, 31))	 return FALSE;
	if (!IKWriteAmp    (Axis, 31))	 return FALSE;

	// Offset = 0
	if (!IKWriteOffset (Axis, 0, 0)) return FALSE;

	return IKStore (Axis);
}


///////////////////////////////////////////////////
// IKRomWrite: Writes Data to Adr of EEPROM on Card
///////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKRomWrite (USHORT Card, BYTE Adr, BYTE Data)
{
	BOOL  Bit;

	if (!PollRom(Card, (BYTE)(0xA0 | ((Adr & 0x0100)>>3 & 0x02))))	return FALSE;		// Wait until EEPROM ready
	if (!Write8I2C (Card, Adr))										return FALSE;		// EEPROM address
	if (!InI2C (Card, &Bit))	  									return FALSE;		// Read acknowledge
	if (!Bit)																	
	{
		if (!Write8I2C (Card, Data)) 								return FALSE;		// Write EEPROM data
		if (!InI2C(Card, &Bit))										return FALSE;		// Read acknowledge
	}
	else return FALSE;
	if (!StopI2C(Card))												return FALSE;		// Stop bit
	return TRUE;
}


///////////////////////////////////////////////////////
// IKRomRead: Reads from Adr of EEPROM on Card to pData
///////////////////////////////////////////////////////
DLLEXPORT BOOL WINAPI IKRomRead (USHORT Card, BYTE Adr, BYTE* pData)
{
	BOOL  Bit;

	if (!PollRom(Card, (BYTE)(0xA0 | ((Adr & 0x0100)>>3 & 0x02))))			return FALSE;	// Wait until EEPROM ready
	if (!Write8I2C (Card, Adr))												return FALSE;	// EEPROM address
	if (!InI2C(Card, &Bit))													return FALSE;	// Read acknowledge
	if (!Bit)																	
	{
		if (!StartI2C (Card))												return FALSE;	// Start bit
		if (!Write8I2C (Card, (BYTE)(0xA1 | ((Adr & 0x0100)>>3 & 0x02))))	return FALSE;	// EEPROM ID, page and read
		if (!InI2C(Card, &Bit)) 											return FALSE;	// Read acknowledge
		if (!Bit)																	
		{
			if (!Read8I2C (Card, pData)) 									return FALSE;	// Read EEPROM data
			if (!OutI2C(Card, TRUE)) 										return FALSE;	// Send no acknowledge
		}
	}
	else return FALSE;
	if (!StopI2C(Card))														return FALSE;	// Stop bit
	return TRUE;
}